/*----------------------------------------------------------------------
 lils.c - Linearizing Integral Least Squares
 by Andy Allinger, 2025, released to the public domain

    Permission  to  use, copy, modify, and distribute this software and
    its documentation  for  any  purpose  and  without  fee  is  hereby
    granted,  without any conditions or restrictions.  This software is
    provided "as is" without express or implied warranty.

 Parameter estimates for models of growth and vibration, specifically:
      Exponential growth
      Diminishing returns / monomolecular model
      Gompertz growth
      Logistic growth
      Korf growth model
      Sinusoid oscillation
      Damped vibration
      Logarithmic spiral

----------------------------------------------------------------------*/
#include <stdbool.h>
#include <float.h>
#include <math.h>
bool dafdiv (double numer, double denom);
int osolvd (int m, int n, double *a, int lda, double *b, double reltol,
             int *krank, double *x, double *w, double *c, int *ind);

/*----------------------------------------------------------------------
dtrnd1 - remove linear trends

___Name___Type______In/Out___Description________________________________
   x[n]   double*   In       Independent variable
   y[n]   double*   Both     Dependent variable
   n      int       In       # points in approximation
----------------------------------------------------------------------*/
void dtrnd1 (double *x, double *y, int n)
{
/* Local variables */
	int i;
	double a0, a1, det, sx, sx2, sxy, sy, x1, x2, y1;
/* Function Body */
	if (n < 2) return;
/* Least-squares sums */
	sx = 0.;
	sx2 = 0.;
	sy = 0.;
	sxy = 0.;
	for (i = 0; i < n; ++i) {
		x1 = x[i];
		y1 = y[i];
		x2 = x1 * x1;
		sx += x1;  /* SUM x */
		sx2 += x2;  /* SUM x^2 */
		sy += y1;  /* SUM y */
		sxy += x1 * y1;  /* SUM x @ y */
	}
/* Solve system of equations. */
	det = n * sx2 - sx * sx;
	a0 = sx2 * sy - sx * sxy;
	a1 = n * sxy - sx * sy;
	if (! dafdiv(a0, det) || ! dafdiv(a1, det)) return;
	a0 /= det;
	a1 /= det;
/* Detrend. */
	for (i = 0; i < n; ++i) y[i] = y[i] - a0 - x[i] * a1;
	return;
} /* end of dtrnd1 */

/*----------------------------------------------------------------------
dtrnd2 - remove second order trend

___Name___Type______In/Out___Description________________________________
   x[n]   double*   In       Independent variable
   y[n]   double*   Both     Dependent variable
   n      int       In       # points in approximation
----------------------------------------------------------------------*/
void dtrnd2 (double *x, double *y, int n)
{
/* Local variables */
	int i;
	double a0, a1, a2, det, sub1, sub2, sub3, sub4, sub5, sub6,
	        sx, sx2, sx3, sx4, sy, sxy, sx2y, x1, x2, y1;
/* Function Body */
	if (n < 3) return;
/* Least-squares sums. */
	sx = 0.;
	sx2 = 0.;
	sx3 = 0.;
	sx4 = 0.;
	sy = 0.;
	sxy = 0.;
	sx2y = 0.;
	for (i = 0; i < n; ++i) {
		x1 = x[i];
		y1 = y[i];
		x2 = x1 * x1;
		sx += x1;  /* SUM x */
		sx2 += x2;  /* SUM x2 */
		sx3 += x2 * x1;  /* SUM x^3 */
		sx4 += x2 * x2;  /* SUM x^4 */
		sy += y1;  /* SUM y */
		sxy += x1 * y1;  /* SUM x @ y */
		sx2y += x2 * y1;  /* SUM x^2 @ y */
	}
/* Parabolic approximation */
	sub1 = sx2 * sx4 - sx3 * sx3;
	sub2 = sx * sx4 - sx2 * sx3;
	sub3 = sx * sx3 - sx2 * sx2;
	sub4 = sxy * sx4 - sx2y * sx3;
	sub5 = sxy * sx3 - sx2y * sx2;
	sub6 = sx2y * sx - sxy * sx2;
	det = n * sub1 - sx * sub2 + sx2 * sub3;
	a0 = sy * sub1 - sx * sub4 + sx2 * sub5;
	a1 = n * sub4 - sy * sub2 + sx2 * sub6;
	a2 = n * (-sub5) - sx * sub6 + sy * sub3;
	if (! dafdiv(a0, det) || ! dafdiv(a1, det) || ! dafdiv(a2, det)) return;
	a0 /= det;
	a1 /= det;
	a2 /= det;
/* Detrend */
	for (i = 0; i < n; ++i) y[i] = y[i] - a0 - x[i] * (a1 + x[i] * a2);
	return;
} /* end of dtrnd2 */

/*----------------------------------------------------------------------
ratest - Estimate the growth rate.
Input must be sorted in order of increasing X.

___Name_____Type______In/Out____Description_____________________________
   x[n]     double*   In        Independent variable (time)
   y[n]     double*   In        Quantity that grows
   n        int       In        Number of data points
   r        double*   Out       Growth rate
   s[n]     double*   Neither   Work array
----------------------------------------------------------------------*/
int ratest (double *x, double *y, int n, double *r, double *s)
{
/* Local variables */
	int j;
	double ts2, tsy;
/* Function Body */
	s[0] = 0.;
	for (j = 1; j < n; ++j) {
		if (x[j] <= x[j-1]) return 2;
		s[j] = s[j-1] + (y[j] + y[j-1]) * 0.5 * (x[j] - x[j-1]);
	}
	dtrnd1 (x, s, n);
	ts2 = 0.;
	tsy = 0.;
	for (j = 0; j < n; ++j) {
		ts2 += s[j] * s[j];
		tsy += s[j] * y[j];
	}
	if (dafdiv(tsy, ts2)) {
		*r = tsy / ts2;
	} else {
		return 1;
	}
	return 0;
} /* end of ratest */

/*----------------------------------------------------------------------
freest - estimate the frequency
Input must be sorted in order of increasing X.

___Name___Type______In/Out____Description_______________________________
   x[n]   double*   In        Independent variable (time)
   y[n]   double*   In        Quantity that oscillates
   n      int       In        Number of data points
   f      double*   Out       Frequency
   s[n]   double*   Neither   Work array
----------------------------------------------------------------------*/
int freest (double *x, double *y, int n, double *f, double *s)
{
/* Constants */
	const double TWOPI = 6.283185307179586477;
/* Local variables */
	int j;
	double sp, spp, ts22, ts2y, w2;
/* Function Body */
	if (n < 4) return 2;
	s[0] = 0.;
	for (j = 1; j < n; ++j) {
		if (x[j] <= x[j-1]) return 4;
		s[j] = s[j-1] + (y[j] + y[j-1]) * 0.5 * (x[j] - x[j-1]);
	}
	sp = 0.;
	for (j = 1; j < n; ++j) {
		spp = sp;
		sp = s[j];
		s[j] = s[j-1] + (s[j] + spp) * 0.5 * (x[j] - x[j-1]);
	}
	dtrnd2 (x, s, n);
	ts22 = 0.;
	ts2y = 0.;
	for (j = 0; j < n; ++j) {
		ts22 += s[j] * s[j];
		ts2y += s[j] * y[j];
	}
	if (dafdiv(ts2y, ts22)) {
		w2 = -ts2y / ts22;
	} else {
		return 1;
	}
	if (w2 < 0.) return 3;
	*f = sqrt(w2) / TWOPI;
	return 0;
} /* end of freest */

/*----------------------------------------------------------------------
korest - Estimate the growth rate according to the Korf model.
Input must be sorted in order of increasing X.

___Name_____Type______In/Out____Description_____________________________
   x[n]     double*   In        Independent variable (time)
   y[n]     double*   In        Quantity that grows
   n        int       In        Number of data points
   r        double*   Out       Growth rate
   lny[n]   double*   Neither   Work array
   s[n]     double*   Neither   Work array
----------------------------------------------------------------------*/
int korest (double *x, double *y, int n, double *r, double *lny, double *s)
{
/* Local variables */
	int j;
	double denom, dn, numer, odx, q, sox,
	        tox, tox2, tsox, toxy, tsox2, ts2ox2, tsoxy, ty;
/* Function Body */
	for (j = 0; j < n; ++j) {
		if (y[j] <= 0.) return 2;
		lny[j] = log(y[j]);
	}
	s[0] = 0.;
	for (j = 1; j < n; ++j) {
		if (x[j] - x[j-1] <= 0.) return 3;
		s[j] = s[j-1] + 0.5 * (lny[j] + lny[j-1]) * (x[j] - x[j-1]);
	}
	ts2ox2 = 0.;  /* Collect sums. */
	tsox2 = 0.;
	tsox = 0.;
	tox = 0.;
	tox2 = 0.;
	tsoxy = 0.;
	toxy = 0.;
	ty = 0.;
	dn = (double) (n);
	for (j = 0; j < n; ++j) {
		odx = 1. / x[j];
		sox = s[j] * odx;
		tox += odx;
		tox2 += odx * odx;
		tsox += sox;
		tsox2 += sox * odx;
		ts2ox2 += sox * sox;
		tsoxy += sox * lny[j];
		toxy += odx * lny[j];
		ty += lny[j];
	}
/* Solve system by Cramer's rule. */
	numer =   tsoxy * (dn * tox2 - tox * tox)
	        - tsox2 * (dn * toxy - tox * ty)
	        + tsox * (tox * toxy - tox2 * ty);
	denom =   ts2ox2 * (dn * tox2 - tox * tox)
	        - tsox2 * (dn * tsox2 - tox * tsox)
	        + tsox * (tox * tsox2 - tox2 * tsox);
	if (dafdiv(numer, denom)) {
		q = numer / denom;
	} else {
		return 1;
	}
	*r = 1. - q;
	return 0;
} /* end of korest */

/*----------------------------------------------------------------------
damest - growth rate and frequency of a damped vibration
Input must be sorted in order of increasing X.

Given a dataset (x,y) and a model

    y = c e^(r x) cos(2 pi f x - p) + k

Determine the parameters r and f.  Amplitude, phase shift, and offset
may then be found be linear regression.

___Name____Type_____In/Out____Description_______________________________
   x[n]    double   In        Independent variable (time)
   y[n]    double   In        Quantity that oscillates
   n       int      In        Number of data points
   r       double   Out       Rate, negative for decay
   f       double   Out       Frequency
   s1[n]   double   Neither   Work array
   s2[n]   double   Neither   Work array
----------------------------------------------------------------------*/
int damest(double *x, double *y, int n, double *r, double *f, double *s1, double *s2)

{
/* Constants */
	const int P = 5;
	const double TWOPI = 6.283185307179586477;
/* Local variables */
	int i, j, k, ind[P], ierr;
	double a[P*P], b[P], c[P], d, dif, tol, soln[P], w[P], w2, x2;
/* Function Body */
	if (n < 5) return 2;
	tol = DBL_EPSILON;  /* Tolerance is machine epsilon. */
	s1[0] = 0.;
	s2[0] = 0.;
	for (j = 1; j < n; ++j) {  /* Integrate. */
		dif = x[j] - x[j-1];
		if (dif <= 0.) return 4;
		s1[j] = s1[j-1] + 0.5 * (y[j] + y[j-1]) * dif;
		s2[j] = s2[j-1] + 0.5 * (s1[j] + s1[j-1]) * dif;
	}
/* Solve the system of linear equations. */
	for (j = 0; j < P; ++j) {
		for (i = 0; i < P; ++i) a[i+j*P] = 0.;
		b[j] = 0.;
	}
	for (j = 0; j < n; ++j) {  /* Collect sums. */
		x2 = x[j] * x[j];
		a[0] += s2[j] * s2[j];
		a[1] += s1[j] * s2[j];
		a[2] += x2 * s2[j];
		a[3] += x[j] * s2[j];
		a[4] += s2[j];
		a[6] += s1[j] * s1[j];
		a[7] += x2 * s1[j];
		a[8] += x[j] * s1[j];
		a[9] += s1[j];
		a[12] += x2 * x2;
		a[13] += x2 * x[j];
		a[14] += x2;
		a[18] += x2;
		a[19] += x[j];
		b[0] += s2[j] * y[j];
		b[1] += s1[j] * y[j];
		b[2] += x2 * y[j];
		b[3] += x[j] * y[j];
		b[4] += y[j];
	}
	for (j = 0; j < P-1; ++j) {  /* Copy to upper triangle. */
		for (i = j+1; i < P; ++i) a[j+i*P] = a[i+j*P];
	}
	a[24] = (double) n;
	ierr = osolvd (P, P, a, P, b, tol, &k, soln, w, c, ind);
	if (ierr != 0) return ierr;
	if (k != P) return 1;
	*r = 0.5 * soln[1];
	d = *r;
	w2 = -(soln[0] + d * d);
	if (w2 < 0.) return 3;
	*f = sqrt(w2) / TWOPI;
	return 0;
} /* end of damest */

/*----------------------------------------------------------------------
spiest - estimate parameters for a logarithmic spiral
Input must be sorted in order of increasing X.

Given a dataset (x,y) and a model

    y = c e^((r + iw)x - ip) + y_0

where x is a real variable and y is complex,
determine the parameters r, f=w/2pi, c, p, and y0.

___Name_______Type______In/Out____Description___________________________
   x[n]       double*   In        Independent variable (time)
   yri[2*n]   double*   In        Coordinates in the complex plane
                                    yri[0:n-1] the real parts
                                    yri[n:2*n-1] the imaginary parts
   n          int       In        Number of data points
   rate       double*   Out       Rate, negative for decay
   freq       double*   Out       Frequency, cycles per unit x
   amp        double*   Out       Amplitude at initial time
   phase      double*   Out       Phase offset, radians
   y0r        double*   Out       Real part of spiral center
   y0i        double*   Out       Imaginary part of spiral center
   w[8*n]     double*   Neither   Work array
----------------------------------------------------------------------*/
int spiest (double *x, double *yri, int n, double *rate, double *freq,
       double *amp, double *phase, double *y0r, double *y0i, double *w)
{
/* Constants */
	const int Q = 4;
	const double TWOPI = 6.283185307179586477;
/* Local variables */
	int ierr, j, k, ind[Q];
	double col[Q], e, imsimy, imsrey, resimy, resrey, soln[Q],
	        t, tol, tsq, work[Q];
/* Function Body */
	if (n < 5) return 2;
	tol = DBL_EPSILON;  /* Tolerance is machine epsilon. >*/
	w[0] = 0.;
	w[n] = 0.;
	for (j = 1; j < n; ++j) {  /* Integrate. */
		t = x[j] - x[j-1];
		if (t <= 0.) return 5;
		w[j] = w[j-1] + 0.5 * (yri[j] + yri[j-1]) * t;
		w[j+n] = w[j-1+n] + 0.5 * (yri[j+n] + yri[j-1+n]) * t;
	}
	dtrnd1 (x, &w[0], n); /* Detrend. */
	dtrnd1 (x, &w[n], n);
	resrey = 0.;
	imsimy = 0.;
	resimy = 0.;
	imsrey = 0.;
	tsq = 0.;
	for (j = 0; j < n; ++j) {
		resrey += w[j] * yri[j];
		imsimy += w[j+n] * yri[j+n];
		resimy += w[j] * yri[j+n];
		imsrey += w[j+n] * yri[j];
		tsq += w[j] * w[j] + w[j+n] * w[j+n];
	}
/* Growth rate and frequency */
	t = resrey + imsimy;
	if (dafdiv(t, tsq)) {
		*rate = t / tsq;
	} else {
		return 3;
	}
	t = resimy - imsrey;
	if (dafdiv(t, tsq)) {
		*freq = t / tsq;  /* angular frequency */
	} else {
		return 4;
	}
/* Solve the system of linear equations. */
	for (j = 0; j < n; ++j) {
		e = exp(*rate * x[j]);
		w[j] = e * cos(*freq * x[j]);
		w[j+n] = e * sin(*freq * x[j]);
		w[j+2*n] = e * sin(*freq * x[j]);
		w[j+3*n] = -e * cos(*freq * x[j]);
		w[j+4*n] = 1.;
		w[j+5*n] = 0.;
		w[j+6*n] = 0.;
		w[j+7*n] = 1.;
	}
	ierr = osolvd (2*n, Q, w, 2*n, yri, tol, &k, soln, work, col, ind);
	if (ierr != 0) return ierr;
	if (k != Q) return 1;
	*freq /= TWOPI;
	*amp = sqrt(soln[0] * soln[0] + soln[1] * soln[1]);
	*phase = atan2(soln[1], soln[0]);
	*y0r = soln[2];
	*y0i = soln[3];
	return 0;
} /* end of spiest */
/* End of lils.c */
