/*----------------------------------------------------------------------
osolvd - solve overconstrained systems of linear equations
algorithm:  complete orthogonal factorization
by Andy Allinger, 2023-2025, released to the public domain

   Permission  to  use, copy, modify, and distribute this software and
   its documentation  for  any  purpose  and  without  fee  is  hereby
   granted,  without any conditions or restrictions.  This software is
   provided "as is" without express or implied warranty.
----------------------------------------------------------------------*/
#include <math.h>

/*----------------------------------------------------------------------
osolvd - linear least squares problem

Solve Ax = b in the least-squares sense by complete orthogonal
factorization.  Factor A:

     PRx = b

Factor R:

     P S^T Q^T x = b

where P and Q have orthogonal columns, and R and S are right
triangular.

The subroutine is intended for the case M .GE. N
It may be used in the case M < N if A is dimensioned A[N,N]

___Name_______Type______In/Out___Description____________________________
   m          int       In       Rows in A
   n          int       In       Columns in A
   a[lda*n]   double*   In       Matrix of size [M,N]
                                      A is destroyed.
   lda        int       In       Row bound of A
                                      LDA >= M if M >= N
                                      LDA >= N if M < N
   b[m]       double*   In       Right-hand vector B[M]
   reltol     double    In       Relative tolerance
   krank      int*      Out      Rank of A
   x[n]       double*   Out      Solution vector
   w[n]       double*   None     Work array
   c[n]       double*   None     Squared column lengths
   ind[n]     int*      None     Which columns to use

Error codes:                                0 = no errors
                                            1 = M out of bounds
                                            2 = N out of bounds
                                            3 = not enough memory
                                            4 = RELTOL out of bounds
                                            5 = Can't determine rank
----------------------------------------------------------------------*/
int osolvd (int m, int n, double *a, int lda, double *b, double reltol,
             int *krank, double *x, double *w, double *c, int *ind)
{
/* Constants */
	const double BIG = 1e36;
	const double BIGSR = 1e18;
	const double PAD = 1e1;
/* Local variables */
	int i, iswap, j, jmax, jp, k, kp, l, ll;
	double d, t, tau, ulen;
/* Parameter adjustments */
	--b;
	--ind;
	--c;
	--w;
	--x;
	a -= lda + 1;
/* Function Body */
	if (m <= 0 || m > lda) return 1;
	if (n <= 0) return 2;
	if (m < n && n > lda) return 3;
	if (reltol <= 0. || reltol >= 1.) return 4;
/* Matrix norm. */
	tau = 0.;
	for (j = 1; j <= n; ++j) {
		c[j] = 0.;
		x[j] = 0.;
		for (i = 1; i <= m; ++i) {
			t = a[i+j*lda];
			c[j] += t * t;
		}
		tau += c[j];
	}
	if (0. == tau) return 0;
	tau = sqrt(tau) * reltol;  /* critical magnitude */
/*----------------------------------------------------------------------
            Factor A -> P R

Matrix P is formed in columns of array A.  Rows of R are stored in X.
As soon as each column of P is complete, compute the next element of
P^T b and store in array W.  Then overwrite the column of P in array A
with the row of R in array X.  Array A now holds a permutation of R^T
----------------------------------------------------------------------*/
	l = 0;
	for (i = 1; i <= n; ++i) ind[i] = i;  /* identity permutation */
	for (kp = 1; kp <= n; ++kp) {
		for (j = 1; j <= n; ++j) x[j] = 0.;  /* Clear x. */
		ulen = -BIG;  /* choose largest column */
		k = kp;  /* initialization */
		jmax = kp;
		for (jp = l+1; jp <= n; ++jp) {
			j = ind[jp];
			if (c[j] > ulen) {
				ulen = c[j];
				k = j;
				jmax = jp;
			}
		}
		ulen = 0.;  /* Recompute magnitude. */
		for (i = 1; i <= m; ++i) {
			t = a[i+k*lda];
			ulen += t * t;
		}
		ulen = sqrt(ulen);
		if (ulen < tau) {
			c[k] = -BIGSR;
			continue;
		}
		++l;  /* Increment rank. */
		iswap = ind[jmax];  /* Swap indices. */
		ind[jmax] = ind[l];
		ind[l] = iswap;
		d = 1. / ulen;  /* Normalize. */
		for (i = 1; i <= m; ++i) a[i+k*lda] *= d;
		x[k] = ulen;
		for (jp = l+1; jp <= n; ++jp) {
			j = ind[jp];
			d = 0.;  /* dot product */
			for (i = 1; i <= m; ++i) d += a[i+j*lda] * a[i+k*lda];
			for (i = 1; i <= m; ++i) a[i+j*lda] -= d * a[i+k*lda];  /* Subtract. */
			x[j] = d;
			c[j] -= d * d;  /* Update magnitude. */
		}
		w[l] = 0.;  /* Multiply P^T b */
		for (i = 1; i <= m; ++i) w[l] += a[i+k*lda] * b[i];
		for (i = 1; i <= n; ++i) a[i+k*lda] = x[i];  /* Replace P with R^T */
	}
/*----------------------------------------------------------------------
            Factor R -> S^T Q^T

Columns of Q are formed in in columns of A.  Rows of S^T are stored in
array X.  As soon as each row of S^T is complete, solve an equation of
the system

            S^T y = P^T b

On completion, array A holds a permutation of matrix Q, and array W
holds vector y.
----------------------------------------------------------------------*/
	tau /= PAD;
	ll = 0;
	for (kp = 1; kp <= l; ++kp) {
		k = ind[kp];
		for (jp = 1; jp <= ll; ++jp) {
			j = ind[jp];
			d = 0.;  /* dot product */
			for (i = 1; i <= n; ++i) d += a[i+j*lda] * a[i+k*lda];
			for (i = 1; i <= n; ++i) a[i+k*lda] -= d * a[i+j*lda];  /* Subtract. */
			x[jp] = d;
		}
		ulen = 0.;  /* vector magnitude */
		for (i = 1; i <= n; ++i) {
			t = a[i+k*lda];
			ulen += t * t;
		}
		ulen = sqrt(ulen);
		if (ulen < tau) return 5;  /* rank slip */
		++ll;
		d = 1. / ulen;  /* Normalize. */
		for (i = 1; i <= n; ++i) a[i+k*lda] *= d;
		x[ll] = ulen;
		for (jp = 1; jp <= ll-1; ++jp) w[ll] -= x[jp] * w[jp];  /* Subtract mid terms. */
		w[ll] /= x[ll];  /* Divide by pivot. */
	}
/* Multiply x = Qy */
	for (i = 1; i <= n; ++i) x[i] = 0.;
	for (jp = 1; jp <= l; ++jp) {
		j = ind[jp];
		for (i = 1; i <= n; ++i) {
			x[i] += a[i+j*lda] * w[jp];
		}
	}
	*krank = l;
	return 0;
} /* end of osolvd */
