/* Fit Ratkowsky's pasture growth data */
#include <stdio.h>
#include <math.h>
void codd (int n, double *y, double *z, double *result);
int ratest (double *x, double *y, int n, double *r, double *s);
int korest (double *x, double *y, int n, double *r, double *lny, double *s);
#define N 9

int main()
{
/* Initialized data */
	const int n = N;
	static double x[N] = { 9.,14.,21.,28.,42.,57.,63.,70.,79. };
	static double y[N] = { 8.93,10.8,18.59,22.33,39.35,56.11,61.73,64.62,
	    67.08 };
	static double b1 = 72.462237576;
	static double b2 = 2.6180768402;
	static double b3 = .067359200066;
 /* Local variables */
 	int ierr, j;
	double a, b, b1est, b2est, b3est, c, det, dn, ody[N], relerr, result,
	         sx, sx2, sxy, sy, t, w0[N], w1[N], z[N];

/*----------------------------------------------------------------------
NIST/ITL StRD
Dataset Name:  Rat42             (Rat42.dat)

Procedure:     Nonlinear Least Squares Regression

Description:   This model and data are an example of fitting
               sigmoidal growth curves taken from Ratkowsky (1983).
               The response variable is pasture yield, and the
               predictor variable is growing time.


Reference:     Ratkowsky, D.A. (1983).
               Nonlinear Regression Modeling.
               New York, NY:  Marcel Dekker, pp. 61 and 88.

Data:          1 Response  (y = pasture yield)
               1 Predictor (x = growing time)
               9 Observations
               Higher Level of Difficulty
               Observed Data

Model:         Exponential Class
               3 Parameters (b1 to b3)

               y = b1 / (1+exp[b2-b3*x])  +  e


          Starting Values                  Certified Values

        Start 1     Start 2           Parameter     Standard Deviation
  b1 =   100         75            7.2462237576E+01  1.7340283401E+00
  b2 =     1          2.5          2.6180768402E+00  8.8295217536E-02
  b3 =     0.1        0.07         6.7359200066E-02  3.4465663377E-03

Residual Sum of Squares:                    8.0565229338E+00
Residual Standard Deviation:                1.1587725499E+00
Degrees of Freedom:                                6
Number of Observations:                            9

Fit the transformed model

     1/y = 1/b1 + exp(b2)/b1 exp(-b3 x)

Thus:
        z:  1/y
        a:  1/b1
        b:  exp(b2)/b1
        c:  -b3

    z = a + b exp(c x)

----------------------------------------------------------------------*/

	printf("...Logistic model...\n");
	for (j = 0; j < n; ++j) ody[j] = 1. / y[j];
	ierr = ratest (x, ody, n, &c, w1);
	printf("ratest returns #%i, estimated c: %f\n", ierr, c);
	sx = 0.;  /* least squares line */
	sy = 0.;
	sx2 = 0.;
	sxy = 0.;
	for (j = 0; j < n; ++j) {
		t = exp(c * x[j]);
		sx += t;
		sy += ody[j];
		sx2 += t * t;
		sxy += t * ody[j];
	}
	dn = (double)n;
	det = dn * sx2 - sx * sx;
	a = (sy * sx2 - sx * sxy) / det;
	b = (sxy * n - sx * sy) / det;
	b1est = 1 / a;
	relerr = (b1est - b1) / b1;
	printf("estimated b1: %f, certified b1: %f, relative error %f\n", b1est, b1, relerr);
	b2est = log(b1est * b);
	relerr = (b2est - b2) / b2;
	printf("estimated b2: %f, certified b2: %f, relative error %f\n", b2est, b2, relerr);
	b3est = -c;
	relerr = (b3est - b3) / b3;
	printf("estimated b3: %f, certified b3: %f, relative error: %f\n", b3est, b3, relerr);
/* Error of the estimate */
	for (j = 0; j < n; ++j) z[j] = b1est / (1. + exp(b2est - b3est * x[j]));  /* integral method */
	codd (n, y, z, &result);
	printf("Integral method coefficient of determination: %f\n", result);
	for (j = 0; j < n; ++j) z[j] = b1 / (1. + exp(b2 - b3 * x[j]));
	codd (n, y, z, &result);
	printf("coefficient of determination, certified parameters: %f\n", result);
/*----------------------------------------------------------------------
            Try Gompertz model.  Transform:

     y = a exp(-b exp(-c x))
     ln(y) = ln(a) - b exp(-c x)

Thus:
       z:  ln(y)
       a':  ln(a)
       b':  -b
       c':  -c

     z = a + b exp(c x)

----------------------------------------------------------------------*/
	printf("\n...Gompertz model...\n");
	for (j = 0; j < n; ++j) ody[j] = log(y[j]);
	ierr = ratest (x, ody, n, &c, w1);
	printf("ratest returns #%i, parameter c: %f\n", ierr, c);
	sx = 0.;  /* least squares line */
	sy = 0.;
	sx2 = 0.;
	sxy = 0.;
	for (j = 0; j < n; ++j) {
		t = exp(c * x[j]);
		sx += t;
		sy += ody[j];
		sx2 += t * t;
		sxy += t * ody[j];
	}
	dn = (double) n;
	det = dn * sx2 - sx * sx;
	a = (sy * sx2 - sx * sxy) / det;
	b = (sxy * n - sx * sy) / det;
	a = exp(a);
	b = -b;
	c = -c;
	printf("a: %f, b: %f\n", a, b);
	for (j = 0; j < n; ++j) z[j] = a * exp(-b * exp(-c * x[j]));
	codd (n, y, z, &result);
	printf("coefficient of determination, Gompertz model: %f\n", result);
/*----------------------------------------------------------------------
            Try Korf model.

     y = a exp(-b x^(-c))

Subroutine KOREST internally applies the transformation
     ln y = (ln a) - b x^(-c)
     z:  ln(y)
     a':  ln(a)
     c':  c
given the original (X,Y) values, and returns growth rate C.

----------------------------------------------------------------------*/
	printf("\n...Korf model...\n");
	ierr = korest (x, y, n, &c, w0, w1);
	printf("korest returns #%i, parameter c: %f\n", ierr, c);
	sx = 0.;  /* least squares line */
	sy = 0.;
	sx2 = 0.;
	sxy = 0.;
	for (j = 0; j < n; ++j) {
		t = pow(x[j], -c);
		sx += t;
		sy += ody[j];
		sx2 += t * t;
		sxy += t * ody[j];
	}
	dn = (double) n;
	det = dn * sx2 - sx * sx;
	a = (sy * sx2 - sx * sxy) / det;
	b = (sxy * dn - sx * sy) / det;
	a = exp(a);
	b = -b;
	printf("a: %f, b: %f\n", a, b);
	for (j = 0; j < n; ++j) z[j] = a * exp(-b * pow(x[j], -c));
	codd (n, y, z, &result);
	printf("coefficient of determination, Korf model: %f\n", result);
	printf("program complete\n");
	return 0;
} /* end of main */
